/*=============================================================================
Copyright  2010 AMETEK Programmable Power. All rights reserved.

FILE NAME:  AmetekDC_example.c

PURPOSE:
   This IVI Driver sample program initializes the power supply, configures
   a channel to supply power, perfoms a few queries, demonstrates how to use
   the triggering sub-system, and displays a message pop-up dialog.

NOTES:
   Developed using LabWindows/CVI 9.0
   
   To run this sample program, do the following:
   1) Create a new project in LabWindows/CVI.
   2) Add this file to the project.  To add files to the project,
      select the Add Files To Project from the Edit menu of the 
      Project window.
   3) Add one of the following files to the project:
      AmetekDC.fp, AmetekDC.c
   4) Check that the resource name used in AmetekDC_InitWithOptions()
      is correct.
   5) Run the project.
   6) If you want to run this sample program and the instrument is not
      present, set the Simulate flag to 1 ( Example: "Simulate=1").
      The default simulation model is an XTR 100-8.5 
      
=============================================================================*/

/*=============================================================================
           This IVI driver also has two special options
       
  1) You can choose a Model to use for simulation purposes. 
     This is done by adding ",DriverSetup=Model: XTR 100-8.5" or another
     model of your choice to the Option String. Include this string right
     after 'Cache=1' inside the existing quotes of the Option String in 
     funciton AmetekDC_InitWithOptions.
      
  2) You can choose a Channel List to use for controlling CANbus 
     (multichannel) instruments via your RS232 or GPIB instrument. This is 
     done by adding ",DriverSetup=Channels: 1-2-4-50" or another set of numbers, 
     to the Option String. Include this string right after 'Cache=1' inside 
     the existing quotes of the Option String in funciton AmetekDC_InitWithOptions
     The valid channel range is 1 to 50.
      
  NOTE 1: Do NOT use commas or spaces to separate the channels in your 
          channel list. You must use the dash (-) character.
            
  NOTE 2: The two options for DriverSetup are mutually exclusive. ie, you
          cannot choose a model for simulation, and specify a channel list. 
          If you choose to simulate and insert a channel list, the driver
          will default to an XTR 100-8.5 on all channels.
   
  NOTE 3: If you do not include a channel list, the driver will create a 
          default channel 'A', which will communicate with the GPIB or 
          Serial device, regardless of its CANbus address. Use this channel
          "A" in all calls to function that require a channel name 
          (as demonstrated below). Do NOT declare this channel yourself in the
          Option String.                   
=============================================================================*/ 

#include <stdio.h>
#include <utility.h>
#include <stdlib.h> 
#include <string.h>
#include "AmetekDC.h"

#ifdef _CVI_
#include <userint.h>
#endif

// Test prototype.
ViStatus fnTestVoltageRange(ViSession DUT,
                                        ViChar channelName[], 
                                        ViReal64 stepSize_Volts, 
                                        ViReal64 tolerance_Volts);

ViStatus fnTestOVPRange(ViSession DUT, 
                                   ViChar channelName[], 
                                   ViReal64 stepSize_Volts);

ViStatus fnTestScpiOperRegEnab(ViSession DUT, 
                                            ViChar channelName[], 
                                            ViStatus (*pfnSetTriggerBit)(ViSession DUT,ViChar channelName[],ViBoolean bBitState),
                                            ViChar ucTriggerBitNumb,
                                            ViUInt32 attrEventCmdId,
                                            ViUInt32 attrEnabCmdId,
                                            ViUInt32 attrPtrCmdId,
                                            ViUInt32 attrNtrCmdId,
                                            ViUInt32 attrSumCondCmdId,
                                            ViChar ucSummaryBitNumb);

ViStatus fnTestSesr(ViSession DUT, 
                        ViChar channelName[]);

ViStatus fnTestAutoSequence(ViSession DUT, 
                        ViChar channelName[]);

ViStatus fnTestFrontPanelTimeout(ViSession DUT,
                        ViChar channelName[]);
                        
ViStatus fnTestCommandDelay(ViSession DUT);

// Utility function prototypes.
ViReal64 fnGetLsd(ViReal64 fModel);
ViStatus fnSetStatOperCshMasterFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperCshSlaveFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperShutSdFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperShutProtFldFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperLocalLockFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperMeasFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

ViStatus fnSetStatOperProgRunningFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState);

/**************************************************************************** 
 *---------------- Instrument Specific Driver Function Declarations -----------------* 
 ****************************************************************************/

/*------- START: Multichannel Functions -----------------------------------------*/

/*------------ General Communication Functions ------------------------------*/
ViStatus _VI_FUNC AmetekDC_SendCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViConstString cmdParameters );
                                             
ViStatus _VI_FUNC AmetekDC_SendBooleanCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViBoolean value);

ViStatus _VI_FUNC AmetekDC_SendInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViInt16 value);

ViStatus _VI_FUNC AmetekDC_SendUInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViUInt16 value);

ViStatus _VI_FUNC AmetekDC_SendInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViInt32 value);

ViStatus _VI_FUNC AmetekDC_SendUInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViUInt32 value);

ViStatus _VI_FUNC AmetekDC_SendReal64Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViReal64 value);

ViStatus _VI_FUNC AmetekDC_SendStringCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,                                             
                                             ViConstString cmdParameters); 



//------------------------- Start example program ---------------------------------------------
int main( void )
{
   ViSession   instr1 = 0;
   ViStatus    error = VI_SUCCESS;
   ViChar      popUpMsg[256];
   ViReal64    measurement1;
   ViReal64    measurement2;
   ViReal64    voltageSetpoint;
   ViBoolean   outputState;
   ViBoolean   OVPenabled;
   ViReal64    OVPlevel;
   ViBoolean   tripped;

   //*****************************************************************************************************************
   /* Use the following block if you are using a GPIB connection, or 
      comment this section out if using RS232 */
      
   //checkErr( AmetekDC_InitWithOptions("GPIB0::1::INSTR", VI_TRUE, VI_TRUE,
   //        "Simulate=0,RangeCheck=1,QueryInstrStatus=1,Cache=1", &instr1 ) );
   
   //checkErr( AmetekDC_SelectRemoteControlSource( instr1, "A", AmetekDC_VAL_GPIB ) );         
   
   checkErr( AmetekDC_InitWithOptions("TCPIP0::10.11.3.1::5025::SOCKET", VI_TRUE, VI_TRUE,
           "Simulate=0,RangeCheck=1,QueryInstrStatus=1,Cache=1,DriverSetup=MCHADDR:31 &Baudrate:9600", &instr1 ) );
   			   
   /* Un-comment this block if you are using an RS232 or USB connections via COM port number.
      For USB connection, the resource name "ASRL#::INSTR" where # is incremented by 1 */ 
   
   //checkErr( AmetekDC_InitWithOptions("ASRL1::INSTR", VI_TRUE, VI_TRUE,
   //        "Simulate=0,RangeCheck=1,QueryInstrStatus=1,Cache=1,DriverSetup=MCHADDR:2 &Baudrate:57600",&instr1 ) ); 

   //*****************************************************************************************************************
   
 /*           1) "DriverSetup=MCHADDR:#"
 *
 *            Optional Fields (must be separated by '&'):
 *            1) "Model: Option Value"  (useful for testing in simulation mode)
 *            2) "Channels: #-#-#"      (available MCH use with RS-485 connected XG's)
 */
    
	// Set the command timeout delay so that we don't have any errors.
	AmetekDC_SetCommandTimeDelay(instr1, (2*AMETEKDC_MIN_CMD_DELAY_SEC));
    
    
//   checkErr( AmetekDC_SetRemoteMode( instr1, AmetekDC_MASTER_CHANNEL, AMETEKDC_VAL_REM ) );        
    AmetekDC_QueryMaxVoltageLevel(instr1, AMETEKDC_MASTER_CHANNEL, 1, &measurement1);
    AmetekDC_QueryMaxCurrentLimit(instr1, AMETEKDC_MASTER_CHANNEL, 1, &measurement2);

   // Test the command delay parameters.
    checkErr( fnTestCommandDelay(instr1));

   // Test the front panel timeout.
//   checkErr( fnTestFrontPanelTimeout(instr1,AMETEKDC_MASTER_CHANNEL));
   
   // Sweep the entire range of voltage checking OVP at each point.
   //checkErr( fnTestOVPRange(instr1, AMETEKDC_MASTER_CHANNEL, 2));   
   
   // Do full sweep of the voltage to determine if device is operating correctly.
   //checkErr( fnTestVoltageRange(instr1, AMETEKDC_MASTER_CHANNEL, 0.2, 0.2));   

   // Test Autosequence Programming Feature.
   // checkErr(fnTestAutoSequence(instr1, AMETEKDC_MASTER_CHANNEL));
 
/*                        
   // Test SESR register.
   checkErr(fnTestSesr(instr1, AMETEKDC_MASTER_CHANNEL));

   // Tested Master: Operation Current Share SCPI Register
   checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperCshSlaveFlag,  
								 eOPER_CSH_SREG_SLAVE,
								 AMETEKDC_VAL_STAT_OPER_CSH_EVENT,
								 AMETEKDC_VAL_STAT_OPER_CSH_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_CSH_PTR,
								 AMETEKDC_VAL_STAT_OPER_CSH_NTR,
								 AMETEKDC_VAL_STAT_OPER_COND,
								 eOPER_STATUS_CSHARE));
								 
   // Tested Slave: Operation Current Share SCPI Register
   checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperCshMasterFlag,  
								 eOPER_CSH_SREG_MASTER,
								 AMETEKDC_VAL_STAT_OPER_CSH_EVENT,
							  	 AMETEKDC_VAL_STAT_OPER_CSH_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_CSH_PTR,
								 AMETEKDC_VAL_STAT_OPER_CSH_NTR,
								 AMETEKDC_VAL_STAT_OPER_COND,
								 eOPER_STATUS_CSHARE));
   
   // Restore the master state. Since the output doesn't behave properly when
   // set as a slave.
   if( !Ivi_Simulating( instr1 ) )
   {
       checkErr( fnSetStatOperCshMasterFlag(instr1, AMETEKDC_MASTER_CHANNEL,  VI_TRUE));
   }        

   // Tested Interlock: Operation Shtudown SCPI Register
   checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperShutSdFlag,  
								 eOPER_SD_STATUS_EXTSD,
								 AMETEKDC_VAL_STAT_OPER_SHUT_EVENT,
								 AMETEKDC_VAL_STAT_OPER_SHUT_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_SHUT_PTR,
								 AMETEKDC_VAL_STAT_OPER_SHUT_NTR,
								 AMETEKDC_VAL_STAT_OPER_COND,
								 eOPER_STATUS_SD));


   // Tested Foldback: Operation Shtudown Protection SCPI Register
   checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperShutProtFldFlag,  
								 eOPER_SD_PROT_SREG_FLD,
								 AMETEKDC_VAL_STAT_OPER_SHUT_PROT_EVENT,
								 AMETEKDC_VAL_STAT_OPER_SHUT_PROT_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_SHUT_PROT_PTR,
								 AMETEKDC_VAL_STAT_OPER_SHUT_PROT_NTR,
								 AMETEKDC_VAL_STAT_OPER_SHUT_COND,
								 eOPER_SD_STATUS_PROT));

   // Tested Local Lockout: Operation SCPI Register
  checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperLocalLockFlag,  
								 eOPER_STATUS_LLOC,
								 AMETEKDC_VAL_STAT_OPER_EVENT,
								 AMETEKDC_VAL_STAT_OPER_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_PTR,
								 AMETEKDC_VAL_STAT_OPER_NTR,
								 AMETEKDC_VAL_STAT_STATUS_BYTE,
								 eSBYTE_OSR));

   // Tested Program Running: Operation SCPI Register
  checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperProgRunningFlag,  
								 eOPER_STATUS_PROG_RUN,
								 AMETEKDC_VAL_STAT_OPER_EVENT,
								 AMETEKDC_VAL_STAT_OPER_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_PTR,
								 AMETEKDC_VAL_STAT_OPER_NTR,
								 AMETEKDC_VAL_STAT_STATUS_BYTE,
								 eSBYTE_OSR));   
   
   // Tested Measurement: Operation SCPI Register
  checkErr( fnTestScpiOperRegEnab(instr1, 
								 AMETEKDC_MASTER_CHANNEL, 
								 fnSetStatOperMeasFlag,  
								 eOPER_STATUS_MEAS,
								 AMETEKDC_VAL_STAT_OPER_EVENT,
								 AMETEKDC_VAL_STAT_OPER_ENABLE,
								 AMETEKDC_VAL_STAT_OPER_PTR,
								 AMETEKDC_VAL_STAT_OPER_NTR,
								 AMETEKDC_VAL_STAT_STATUS_BYTE,
								 eSBYTE_OSR));    
								 
								 */
   
   /* --- General Output and Protection Configuration ----------------------- */
                    
   checkErr( AmetekDC_ConfigureOutputEnabled( instr1, AMETEKDC_MASTER_CHANNEL, VI_TRUE ) );
   checkErr( AmetekDC_ConfigureVoltageLevel( instr1,AMETEKDC_MASTER_CHANNEL, 5.0 ) );      
   checkErr( AmetekDC_ConfigureCurrentLimit( instr1, AMETEKDC_MASTER_CHANNEL, 
                                           AMETEKDC_VAL_CURRENT_REGULATE, 
                                           5.0 ) );          
   
   checkErr( AmetekDC_ConfigureOVP( instr1, AMETEKDC_MASTER_CHANNEL, VI_TRUE, 7.0 ) );
   checkErr( AmetekDC_ConfigureProtectionSetpoint( instr1, AMETEKDC_MASTER_CHANNEL, 
                                                 AMETEKDC_ATTR_OCP_LIMIT,
                                                 AMETEKDC_VAL_PROTECTION_TRIP,
                                                 8.5 ) );      


   // --- Measurements and Instrument Queries ------------------------------- */
   Delay( 1.0 ); /* Delay to ensure output settles before taking measuremnt */ 

   checkErr( AmetekDC_Measure( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_VAL_MEASURE_VOLTAGE, &measurement1 ) );                              
//   checkErr( AmetekDC_Measure( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_VAL_MEASURE_CURRENT, &measurement2 ) );                              
   Delay( 0.1 );
   
   checkErr( AmetekDC_QuerySetpoint( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_ATTR_VOLTAGE_LEVEL, &voltageSetpoint ) );                     
   
//   checkErr( AmetekDC_QueryProtectionTripped( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_VAL_OVP, &tripped ) );    
   //checkErr( AmetekDC_QueryProtectionSettings( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_ATTR_OVP_LIMIT, &OVPenabled, &OVPlevel ) );     
  
   checkErr( AmetekDC_QueryOutputState( instr1, AMETEKDC_MASTER_CHANNEL, AMETEKDC_VAL_OUTPUT_UNDER_VOLTAGE, &outputState ) );                                   

   
   /* --- Print general message for success --------------------------------- */
   sprintf( popUpMsg, "If you are NOT running in Simulation mode, the power\n"
                   "supply will now be configured to a specified settings." );
      
#ifdef _CVI_
   MessagePopup( "Message", popUpMsg );
#else
   printf( "Message:\n%s", popUpMsg );
#endif


Error:
   if( error != VI_SUCCESS )
   {
      ViChar errStr[ 2048 ];

      AmetekDC_GetError( instr1, &error, 2048, errStr );
      
#ifdef _CVI_
      MessagePopup( "Error!", errStr );
#else
      printf( "Error!:\n%s", errStr );
#endif

   }
        
   if( instr1 > 0 )
   {
      AmetekDC_close( instr1 );
   }
   
   return( 0 );
}

/*****************************************************************************
 * Function: fnTestCommandDelay                                  
 * Purpose:  This function does a test of the configuration of the front
 *           panel timeout.
 *
 *****************************************************************************/
ViStatus fnTestCommandDelay(ViSession DUT)
{
    ViStatus       error = VI_SUCCESS;
	ViBoolean bTestFailedState = VI_FALSE;
	ViChar tmpBuf[BUFFER_SIZE] = "";
	ViReal64   driverDelay, readbackDelay;

	// Read current timeout.
    checkErr( AmetekDC_GetCommandTimeDelay(DUT, &driverDelay));
	
	// Set he time delay. - Generally speaking we should get an error here 
	// from setting such a short delay time so long as the readback is enabled.
	// hence no error check on this command.
	AmetekDC_SetCommandTimeDelay(DUT, AMETEKDC_MIN_CMD_DELAY_SEC);

	// Read back and verify that delay was changed.
	checkErr( AmetekDC_GetCommandTimeDelay(DUT, &readbackDelay));
	
	if( readbackDelay != AMETEKDC_MIN_CMD_DELAY_SEC )
	{
       	// Throw error as the read back value is out of range.
        sprintf( tmpBuf, "FrontPanel Timeout Test Failed: should have read 1000 but got %d\n", readbackDelay );
					
       	bTestFailedState = VI_TRUE;
	    goto Failed;	
	}
	
Failed:	
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	
Error:
   AmetekDC_SetCommandTimeDelay(DUT, driverDelay);
   return( error );
}


/*****************************************************************************
 * Function: fnTestFrontPanelTimeout                                  
 * Purpose:  This function does a test of the configuration of the front
 *           panel timeout.
 *
 *****************************************************************************/
ViStatus fnTestFrontPanelTimeout(ViSession DUT, 
                        ViChar channelName[])
{
    ViStatus       error = VI_SUCCESS;
	ViBoolean bTestFailedState = VI_FALSE;
	ViChar tmpBuf[BUFFER_SIZE] = "";
	ViInt32	   frontPanelTimeout, readbackFpTimeout;

   // Check the front panel settings.
   checkErr( AmetekDC_QueryFrontPanelTimeout(DUT, AMETEKDC_MASTER_CHANNEL, &frontPanelTimeout ) );	
   checkErr( AmetekDC_ConfigureFrontPanelTimeout(DUT, AMETEKDC_MASTER_CHANNEL, 1000 ) );	
   checkErr( AmetekDC_QueryFrontPanelTimeout(DUT, AMETEKDC_MASTER_CHANNEL, &readbackFpTimeout ) );
   
   // Check the voltage readback.
   if( readbackFpTimeout != 1000)
   {
       	// Throw error as the read back value is out of range.
        sprintf( tmpBuf, "FrontPanel Timeout Test Failed: should have read 1000 but got %d\n", readbackFpTimeout );
					
       	bTestFailedState = VI_TRUE;
	    goto Failed;	                 	
   }

Failed:	
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	
Error:
   AmetekDC_ConfigureFrontPanelTimeout(DUT, AMETEKDC_MASTER_CHANNEL, frontPanelTimeout );	
   return( error );
}


/*****************************************************************************
 * Function: fnTestVoltageRange                                  
 * Purpose:  This function does a test of the output voltage across the entire
 *           range of the model.  The stepSize indicates the granularity that 
 *           should be applied to the test.  The tolerance indicates how much
 *           deviation can be seen on the output before a failure of the test
 *           will be declared.
 *
 * NOTE: This function will enable the output and set voltages all the way up
 *       to the maximum model voltage.  Users of the function should take care
 *       to not have any sensative loads attached to the output during this test
 *       as it may result in damage to the load.
 *
 *****************************************************************************/
ViStatus fnTestVoltageRange(ViSession DUT, 
                        ViChar channelName[], 
                        ViReal64 stepSize_Volts,
                        ViReal64 tolerance_Volts)
{
    ViStatus       error = VI_SUCCESS;
	ViBoolean bTestFailedState = VI_FALSE;
	ViReal64 voltageModel_Volts = 0;
	ViReal64 voltSetPt_Volts = 0;
	ViReal64 voltMeas_Volts = 0;
	ViInt32  numbSteps = 1; 
	ViChar tmpBuf[BUFFER_SIZE] = "";
	ViInt32 i;
	ViReal64 setPointReadback = 0;
	ViBoolean outputState = VI_FALSE;

       // Lock the session so that the the query status can be checked.
      checkErr( Ivi_LockSession( DUT, VI_NULL ) );
	   
	// Get the voltage models from the DUT.
	//checkErr( AmetekDC_QueryID(DUT, AmetekDC_MASTER_CHANNEL, BUFFER_SIZE, tmpBuf ));
	//AmetekDC_ATTR_ID_QUERY_RESPONSE
	checkErr( Ivi_GetAttributeViString( DUT, VI_NULL, AMETEKDC_ATTR_INSTRUMENT_MODEL, 0, BUFFER_SIZE, tmpBuf ) );
	sscanf(tmpBuf, "%*s %lf%*[^\n]", &voltageModel_Volts);
	
	// Make sure OVP is not an issue with setting the voltage and won't trip during the test.
	checkErr( AmetekDC_ConfigureOVP( DUT, channelName, VI_TRUE, (voltageModel_Volts*1.05) ) );  
	
	// Parse the voltage model from the model string.
	numbSteps = voltageModel_Volts/stepSize_Volts;
	
	// Reset unit to restore initial conditions.
	checkErr( AmetekDC_ResetMultichannelPowerSupply( DUT, channelName) );

	// Clear any errors.
	checkErr( AmetekDC_ClearStatus( DUT, channelName));
	
	
       // Set maximum current for  the smallest current models.  Should prevent slewing
       // due to falling into CC mode.
       checkErr( AmetekDC_ConfigureCurrentLimit( DUT, channelName, 
                                           AMETEKDC_VAL_CURRENT_REGULATE, 
                                           1.4 ) );   
        checkErr( AmetekDC_QuerySetpoint( DUT, channelName, AMETEKDC_ATTR_CURRENT_LIMIT,&setPointReadback ));

	 if(setPointReadback != 1.4)
	 {
       		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "Voltage Range Test Failed: Failure to set the current set point: Iset =  %lf, Iset_read = %lf.\n", \
	                       voltSetPt_Volts, setPointReadback);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;	
	 }	
	 
       // Enabled output to begin test.                                           
	checkErr( AmetekDC_ConfigureOutputEnabled( DUT, channelName, VI_TRUE ) );

      // Verify the output is enabled.
	checkErr( AmetekDC_QueryEnabled( DUT, channelName, AMETEKDC_ATTR_OUTPUT_ENABLED,  &outputState));
	if( outputState == VI_FALSE )
	{
		// Error failure to enable output.
      		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
            sprintf( tmpBuf, "Voltage Range Test Failed: Failure to enable the output.\n");
				
            	bTestFailedState = VI_TRUE;
		goto Failed;			
	}
	
	for(i=0; i < numbSteps; i++)
	{
        // Update the voltage setting.
        voltSetPt_Volts += stepSize_Volts;		
        checkErr( AmetekDC_ConfigureVoltageLevel( DUT, channelName, voltSetPt_Volts ) );  
        checkErr( AmetekDC_QuerySetpoint( DUT, channelName, AMETEKDC_ATTR_VOLTAGE_LEVEL,&setPointReadback ));

	  if(voltSetPt_Volts != setPointReadback)
	  {
       		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "Voltage Range Test Failed: Failure to set the voltage set point: Vset =  %lf, Vset_read = %lf.\n", \
	                       voltSetPt_Volts, setPointReadback);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;	
	  }
		// Measure the 
        checkErr( AmetekDC_Measure( DUT, channelName, AMETEKDC_VAL_MEASURE_VOLTAGE, 
                             &voltMeas_Volts ) );  

        // Check the voltage readback.
        if(( voltMeas_Volts >= (voltSetPt_Volts + voltSetPt_Volts)) || 
           (voltMeas_Volts < (voltSetPt_Volts - voltSetPt_Volts)))
        {
        	// Throw error as the read back value is out of range.
            sprintf( tmpBuf, "Voltage Range Test Failed on step %d at voltage %f\n", i, voltMeas_Volts );
					
            	bTestFailedState = VI_TRUE;
		goto Failed;	                 	
        }
	}

	// Check the driver range functionality to make sure this is operating correctly. This test is 
	// intentionally made to fail.  The driver should prevent the range from being set above the model
	// voltage.
	if(AmetekDC_ConfigureOutputRange(DUT, channelName, AMETEKDC_VAL_RANGE_VOLTAGE, 1000.0) == VI_SUCCESS)
	{
        	// Throw error as the read back value is out of range.
            sprintf( tmpBuf, "Voltage Range Test Failed: Configured the output range to be 1000.0 volts which should be possible.\n");
					
            	bTestFailedState = VI_TRUE;
		goto Failed;	                 	
	}
	
Failed:	
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	
Error:
   Ivi_UnlockSession( DUT, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: fnTestOVP                                  
 * Purpose:  This function does a test of the OVP at various voltages.  The
 *           intent is to verify that the calibration and the system works 
 *           correctly at all voltage ranges.
 *
 * NOTE: Users of the function should take care
 *       to not have any sensative loads attached to the output during this test
 *       as it may result in damage to the load.
 *
 *****************************************************************************/
ViStatus fnTestOVPRange(ViSession DUT, 
                        ViChar channelName[], 
                        ViReal64 stepSize_Volts)
{
    const ViReal64 kfOvpToVsetMargin	= 1.05;
	ViStatus       error = VI_SUCCESS;
	ViInt32 errCode = 0;
	ViInt32 voltageModel_Volts = 0;
	ViReal64 currentModel_Amps = 0;
	ViReal64 voltSetPt_Volts = 0.5;		// Set to largest minimum value for all models
	ViInt32  numbSteps = 1; 
	ViChar tmpBuf[BUFFER_SIZE] = "";
	ViInt32 i;
	ViBoolean queryStatusState = 1;
	ViReal64 fLeastSigDigit = 0.1;
	ViBoolean bTestFailedState = VI_FALSE;

    // Lock the session so that the the query status can be checked.
    checkErr( Ivi_LockSession( DUT, VI_NULL ) );
   
    // Back up the query and enable it.
    queryStatusState = Ivi_QueryInstrStatus(DUT);
	AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, VI_TRUE);

	// Get the voltage models from the DUT.
	checkErr( Ivi_GetAttributeViString( DUT, VI_NULL, AMETEKDC_ATTR_INSTRUMENT_MODEL, 0, BUFFER_SIZE, tmpBuf ) );
	sscanf(tmpBuf, "%*s %d%*[^\n]", &voltageModel_Volts);
	sscanf(tmpBuf, "%*s %*d-%lf%*[^\n]", &currentModel_Amps);
	
	// Get the least significant digit for this voltage model.
	fLeastSigDigit = fnGetLsd(voltageModel_Volts);

	// Parse the voltage model from the model string. Note the minimum value for OVP.
	numbSteps = (voltageModel_Volts - voltSetPt_Volts) / stepSize_Volts;

	// Need to consider setting the initial value to be close or slightly less than 
	// the minimum value. The minimum OVP set point is in the worst case
	// 8.33% of the model voltage.  So we start the Vset point to be kfOvpToVsetMargin
	// below this value so that the OVP is at its minimum possible level for all models.
	voltSetPt_Volts = (voltageModel_Volts * 0.0834) / kfOvpToVsetMargin;
	
	// Reset unit to restore initial conditions.
	checkErr( AmetekDC_ResetMultichannelPowerSupply( DUT, channelName) );
	
	// Clear any errors.
	checkErr( AmetekDC_ClearStatus( DUT, channelName)); 

    // Set maximum current for  the current model.  This should prevent slewing
    // due to falling into CC mode.
    checkErr( AmetekDC_ConfigureCurrentLimit( DUT, channelName, 
                                           AMETEKDC_VAL_CURRENT_REGULATE, 
                                           currentModel_Amps ) ); 
	
    // Enabled output to begin test.                                           
	checkErr( AmetekDC_ConfigureOutputEnabled( DUT, channelName, VI_TRUE ) );
		
	// Start test.
	for(i=0; i < numbSteps; i++)
	{
		// Check correct min OVP setting based on Vset_point.

		// Set OVP set point to 105% of Vset and query for errors.
            if(AmetekDC_ConfigureOVP( DUT,channelName, VI_TRUE, (voltSetPt_Volts * kfOvpToVsetMargin) ) != VI_SUCCESS)
            	{
            		// Found issue, unable to configure the OVP set point. XG has logged the following errors:
            		AmetekDC_GetError( DUT, &error, BUFFER_SIZE, tmpBuf);

            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "OVP Test Failed: Failure to configure the set OVPset to %lf on iteration %d.\n %s\n", \
	                       (voltSetPt_Volts * kfOvpToVsetMargin), i, tmpBuf);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;					

            	}

		
		// Set the voltage. Query for errors. There should be none.
            if(AmetekDC_ConfigureVoltageLevel( DUT,channelName, voltSetPt_Volts ) != VI_SUCCESS)
            	{
            		// Found issue, unable to configure the OVP set point. XG has logged the following errors:
            		AmetekDC_GetError( DUT, &error, BUFFER_SIZE, tmpBuf);

            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "OVP Test Failed: Failure to configure the set Vset to %lf on iteration %d.\n %s\n", \
	                       voltSetPt_Volts, i, tmpBuf);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
            	}

		// Set OVP set point to 105% of Vset and query for errors.
            if(AmetekDC_ConfigureOVP( DUT,channelName, VI_TRUE, ((voltSetPt_Volts * kfOvpToVsetMargin) - (2*fLeastSigDigit)) ) == VI_SUCCESS)
            	{
            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "OVP Test Failed: The protection that prevents the user from setting the OVP \
	                       set point to be no less 105%% of Vset failed. Vset = %lf, OVPset = %lf\n", \
	                       voltSetPt_Volts, ((voltSetPt_Volts * kfOvpToVsetMargin) - fLeastSigDigit) );
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
            	}
		else
		{
			// Verify that the error is a settings conflict error.
        		AmetekDC_GetError( DUT, &error, BUFFER_SIZE, tmpBuf);

			// Query the device for error.
			AmetekDC_error_query(DUT, &errCode, tmpBuf);
			
			// Check to see the error number is a settings conflict error.
			if( errCode != -221)
			{
				// Incorrect error thrown.
	      		      sprintf( tmpBuf, "OVP Test Failed: The protection through an unexpected error when setting the OVP \
	                              set point to be no less 105%% of Vset failed. Vset = %lf, OVPset = %lf, error = %d,%s\n", \
	                              voltSetPt_Volts, ((voltSetPt_Volts * kfOvpToVsetMargin) - fLeastSigDigit), errCode, tmpBuf);
					
      		        	bTestFailedState = VI_TRUE;
				goto Failed;				
			}
			
			// Ignore error as it indicates that the unit has an error.
			error = VI_SUCCESS;
		}


		// Update the voltage setting.
            voltSetPt_Volts += stepSize_Volts;
	}
	
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  

Error:
	// Reset unit to restore initial conditions.
	AmetekDC_ResetMultichannelPowerSupply( DUT, channelName);
	
   // Restore the status checking to value previously set before entering this test.
   AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, queryStatusState);   
   Ivi_UnlockSession( DUT, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: fnGetLsd                                  
 * Purpose:  This function finds the Least significant digit for the model being
 *              used. 
 *
 *****************************************************************************/
ViReal64 fnGetLsd(ViReal64 fModel)
{
	
	// Find the position of the Least Significant Digit of the XG being tested.
	if( ((ViInt16)fModel/100) > 0)
	{
		return 0.1;
	}
	else if (((ViInt16)fModel/10) > 0)
	{
		return 0.01;
	}
	else if ((ViInt16)fModel > 0)
	{
		return 0.001;
	}
	else
	{
		return -1;
	}
}


/*****************************************************************************
 * Function: fnTestOVP                                  
 * Purpose:  This function does verification of an a bit in the operation
 *               status register.  It check to see that the sumary bit function
 *               correctly and that the positive and negative transition registers
 *               function correctly as well as the enable and event registers.
 *               The summary bit is checked in the summary register, thus 
 *               implicitely verifying the condition register of the summary register.
 *  
 *               This function is purely the abstract for a larger test regime. 
 *
 *  Parameters:
 *                 DUT - Vi session handle, pretty self explanatory.
 *     channelName - Address of the DUT.
 * pfnSetTriggerBit - Function pointer that will allow control over the bit that
 *                          is being tested.  The function must be able to cause the
 *                          bit to be set (trigger alarm) and cleared (resolve condition
 *                          that caused the alarm).
 * ucTriggerBitNumb - The bit position of the condition being triggered.  As with
 *                           all register bit position counting starts from zero.
 *        attrPtrCmdId - Command ID for setting the PTR register. 
 *        attrNtrCmdId - Command ID for setting the NTR register.
 *      attrEnabCmdId - Command ID for setting the enable register.
 *     attrEventCmdId - Command ID for querying the event register.
 * attrSumCondCmdId - Command ID for querying the condition of the summary register.
 * ucSummaryBitNumb - The bit position of the summary bit in the summary condition 
 *                               register.
 *
 * NOTE: This function will enable the output and set voltages all the way up
 *       to the maximum model voltage.  Users of the function should take care
 *       to not have any sensative loads attached to the output during this test
 *       as it may result in damage to the load.
 *****************************************************************************/
ViStatus fnTestScpiOperRegEnab(ViSession DUT, 
                        ViChar channelName[], 
                        ViStatus (*pfnSetTriggerBit)(ViSession DUT,ViChar channelName[],ViBoolean bBitState),
                        ViChar ucTriggerBitNumb,
                        ViUInt32 attrEventCmdId,
                        ViUInt32 attrEnabCmdId,
                        ViUInt32 attrPtrCmdId,
                        ViUInt32 attrNtrCmdId,
                        ViUInt32 attrSumCondCmdId,
                        ViChar ucSummaryBitNumb)
{
    ViStatus       error = VI_SUCCESS;
    ViChar errorMsgBuf[BUFFER_SIZE] = "";
    ViChar tmpBuf[BUFFER_SIZE] = "";	
    ViBoolean bTestFailedState = VI_FALSE;	
    ViUInt16 uiRegValue = 0;
    ViUInt16 uiTriggerMask = (1<<ucTriggerBitNumb);
    ViUInt16 uiSumCondMask = (1<<ucSummaryBitNumb);

	
    //************* Positive Test of PTR, NTR, and Summery Bit ***************
    // Setup the register to have its transitions states setup so that the summary bit can be
    // checked.
    AmetekDC_SetRegister(DUT, channelName, attrPtrCmdId, uiTriggerMask);
    AmetekDC_SetRegister(DUT, channelName, attrNtrCmdId, uiTriggerMask);
    AmetekDC_SetRegister(DUT, channelName, attrEnabCmdId, uiTriggerMask);	

    // Set trigger bit which should force the summary bit to detect another event and affect
    // summary condition register.
    checkErr( pfnSetTriggerBit(DUT, channelName,VI_TRUE));

    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrSumCondCmdId, &uiRegValue);

    // Enable register is currently set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiSumCondMask == 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg Sum Enable Test Failed: Summary Condition should have bit %d set: SumCondReg = %d\n", 
                ucSummaryBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;				
    }

    // Read the event register and verify that its bit clears.
    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask == 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                 "SCPI Oper Reg Sum Enable Test Failed: Event register should have bit %d set: EventReg = %d\n", 
                 ucTriggerBitNumb, 
                 uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;				
    }	

    // Confirm the event cleared by reading.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask != 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg Sum Enable Test Failed: Event register should have bit %d cleared: EventReg = %d\n", 
                ucTriggerBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	    
    }	

    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrSumCondCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiSumCondMask != 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg Sum Enable Test Failed: Summary Condition register should have bit %d cleared: SumCondReg = %d\n",
                ucSummaryBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	
    }
	
    // Clear the bit which should force the summary bit to detect another event and affect
    // summary condition register. Since the NTR trigger flag is set this should set the bit
    // in the event register.  Since the bit is also enabled it will set the summary bit as well.
    checkErr(pfnSetTriggerBit(DUT, channelName,VI_FALSE));

    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrSumCondCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiSumCondMask == 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg Sum Enable Test Failed: Summary Condition should have bit %d set: SumCondReg = %d\n", 
                ucSummaryBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	
    }

    //************* Negative Test of NTR ***************	
    // Setup the register to have its transitions states setup so that the summary bit can be
    // checked.
    AmetekDC_SetRegister(DUT, channelName, attrPtrCmdId, uiTriggerMask);
    AmetekDC_SetRegister(DUT, channelName, attrNtrCmdId, 0);

    // Make sure the event register is initially clear.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Set trigger bit which should force the summary bit to detect another event and affect
    // summary condition register.
    checkErr(pfnSetTriggerBit(DUT, channelName,VI_TRUE));

    // Read the event register and verify that its bit clears.
    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask == 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg NTR Test Failed: Event register should have bit %d should be set: EventReg = %d\n",
                ucTriggerBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	
    }	

    // Clear trigger bit which should force the summary bit to detect another event and affect
    // summary condition register.
    checkErr(pfnSetTriggerBit(DUT, channelName,VI_FALSE));
 
    // Read the event register and verify that its bit clears.
    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask != 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg NTR Test Failed: Event register should have bit %d should be cleared: EventReg = %d\n",
                ucTriggerBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	    
    }		


    //************* Negative Test of PTR ***************	
    // Setup the register to have its transitions states setup so that the summary bit can be
    // checked.
    AmetekDC_SetRegister(DUT, channelName, attrPtrCmdId, 0);
    AmetekDC_SetRegister(DUT, channelName, attrNtrCmdId, uiTriggerMask);

    // Make sure the event register is initially clear.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);
	
    // Set trigger bit which should force the summary bit to detect another event and affect
    // summary condition register.
    checkErr(pfnSetTriggerBit(DUT, channelName,VI_TRUE));

    // Read the event register and verify that its bit clears.
    // Read back the summary condition register to make sure the signal was propagated up.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask != 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg PTR Test Failed: Event register should have bit %d should be cleared: EventReg = %d\n",
                ucTriggerBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	
    }	

    // Clear trigger bit which should force the summary bit to detect another event and affect
    // summary condition register.
    checkErr(pfnSetTriggerBit(DUT, channelName,VI_FALSE));
 
    // Read the event register and verify that its bit set as the NTR bit is set/enabled.
    AmetekDC_QueryRegister(DUT, channelName, attrEventCmdId, &uiRegValue);

    // Enable register is current set for this bit so summary condition should be set in summary
    // register.
    if( uiRegValue & uiTriggerMask == 0)
    {
    	// Failure, the summery bit should have been set by the trigger.
      sprintf( errorMsgBuf, 
                "SCPI Oper Reg PTR Test Failed: Event register should have bit %d should be set: EventReg = %d\n",
                ucTriggerBitNumb, 
                uiRegValue);
					
	bTestFailedState = VI_TRUE;
	goto Failed;	
    }		
    	
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", errorMsgBuf );
#else
		      printf( "%s", errorMsgBuf );
#endif   	
	}   	
	

Error:
   return( error );
}


/*****************************************************************************
 * Function: fnSetStatOperCshMasterFlag                                  
 * Purpose:  This function sets the SCPI operation status register, CSH sub register 
 *              master flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.
 *
 * NOTE: This function will also affect the current share slave flag as well.
 *
 *****************************************************************************/
ViStatus fnSetStatOperCshMasterFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;

   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	if(bBitState == VI_TRUE)
	{
      		AmetekDC_SendStringCmd (DUT, io, channelName, AMETEKDC_VAL_SOUR_COMB_CSH, "MAST");
	}
	else
	{
      		AmetekDC_SendStringCmd (DUT, io, channelName, AMETEKDC_VAL_SOUR_COMB_CSH, "SLAV");	  
	}
   }
	

Error:
   return( error );
}


/*****************************************************************************
 * Function: fnSetStatOperCshSlaveFlag                                  
 * Purpose:  This function sets the SCPI operation status register, CSH sub register 
 *              slave flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.
 *
 * NOTE: This function will also affect the current share master flag as well.
 *
 *****************************************************************************/
ViStatus fnSetStatOperCshSlaveFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
     
    // Select which sense to set the bit to.
    if(bBitState != VI_TRUE)
    {
	checkErr(AmetekDC_SetCurrentShareMode(DUT, channelName, "MAST"));
    }
    else
    {
	checkErr(AmetekDC_SetCurrentShareMode(DUT, channelName, "SLAV"));	  
    }
	

Error:
   return( error );
}



/*****************************************************************************
 * Function: fnSetStatOperShutSdFlag                                  
 * Purpose:  This function sets the SCPI operation status register, SHUT sub register 
 *              interlock flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers. The output protection is
 *              cleared after each flag is set so that the settings conflict issues don't 
 *              turn up in the testing code.
 *
 * Note: This function assumes no connection or input to the external shutdown pin.
 *          control is done through modification of the polarity.
 
 *****************************************************************************/
ViStatus fnSetStatOperShutSdFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
                                              
   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	// Select which sense to set the bit to.
	if(bBitState == VI_TRUE)
	{
      		AmetekDC_SendStringCmd (DUT, io, channelName, AMETEKDC_VAL_OUTP_SD_POL, "HIGH");
	}
	else
	{
      		AmetekDC_SendStringCmd (DUT, io, channelName, AMETEKDC_VAL_OUTP_SD_POL, "LOW");	  
	}

	// For the purpose of this situation, the output protection should be cleared so alarms are tripped.
	checkErr(AmetekDC_ResetOutputProtection(DUT, channelName));
   }
	
Error:
   return( error );
}

/*****************************************************************************
 * Function: fnSetStatOperShutProtFldFlag                                  
 * Purpose:  This function sets the SCPI operation status register, PROT sub register 
 *              foldback flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.  In addition this function 
 *              enables and disables the output while also causing changes to the set points.
 *
 * Note: This enables and disables the output, it should not be used with a load attached
 *         as the load could be damaged or the resulting flag might not be set!
 *
 *****************************************************************************/
ViStatus fnSetStatOperShutProtFldFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
                                              
   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	// Select which sense to set the bit to.
	if(bBitState == VI_TRUE)
	{
		// Cause OVP to be tripped by configuring the foldback to occur at a minimum
		// amount of time and to trip on CV mode.
		checkErr(AmetekDC_ConfigureFoldbackSetting( DUT, channelName, AMETEKDC_VAL_FOLD_CV));
		checkErr(AmetekDC_ConfigureFoldDelay( DUT, channelName, AMETEKDC_FOLDBACK_MIN_DELAY_SEC ));
		checkErr( AmetekDC_ConfigureOutputEnabled( DUT, channelName, VI_TRUE ) );

		// Delay the amount of time noted in the delay time.
      		Delay(AMETEKDC_FOLDBACK_MIN_DELAY_SEC);
	}
	else
	{
		// Clear the alarm condition by disabling the foldback.
      		checkErr(AmetekDC_ConfigureFoldbackSetting( DUT, channelName, AMETEKDC_VAL_FOLD_NONE));
      		checkErr(AmetekDC_ResetOutputProtection(DUT, channelName));	  		
	}  
   }
	

Error:
   return( error );
}

	
/*****************************************************************************
 * Function: fnSetStatOperLocalLockFlag                                  
 * Purpose:  This function sets the SCPI operation status register, local lockout
 *              flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.  In addition this function 
 *              enables and disables the output while also causing changes to the set points.
 *
 * Note: This enables and disables the output, it should not be used with a load attached
 *         as the load could be damaged or the resulting flag might not be set!
 *
 *****************************************************************************/
ViStatus fnSetStatOperLocalLockFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
                                              
   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	// Select which sense to set the bit to.
	if(bBitState == VI_TRUE)
	{
		// Set the local lockout flag by making the mode remote.
		checkErr(AmetekDC_SetRemoteMode( DUT, channelName, AMETEKDC_VAL_REM));
	}
	else
	{
		// Clear the local lockout flag by making the mode local.
		checkErr(AmetekDC_SetRemoteMode( DUT, channelName, AMETEKDC_VAL_LOC));
	}  
   }
	

Error:
   return( error );
}


/*****************************************************************************
 * Function: fnSetStatOperProgRunningFlag                                  
 * Purpose:  This function sets the SCPI operation status register, program running
 *              flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.  In addition this function 
 *              enables and disables the output while also causing changes to the set points.
 *
 * Note: This function modifies the autosequence dwell time. Users should not this 
 *         impact and save the value of the dwell time before calling the funciton. 
 * Note: The user should also be aware that the running flag will only be set for the
 *         maximum duration of the dwell time.  So testing for the flag shouldn't be 
 *         put off for too long when using this function. (Currently max dwell is 180s)
 *****************************************************************************/
ViStatus fnSetStatOperProgRunningFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
    ViBoolean queryStatusState = VI_TRUE;	
	
    // Back up the query and enable it.
    queryStatusState = Ivi_QueryInstrStatus(DUT);
                                              
   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	// Select which sense to set the bit to.
	if(bBitState == VI_TRUE)
	{
		// Set the local lockout flag by making the mode remote.
		checkErr(AmetekDC_SetAutoSequenceDwellTime(DUT, channelName, AMETEKDC_MAX_DWELL_TIME_SEC));

		// Set the voltage. Query for errors. There should be none.
		AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, VI_FALSE);
		
		// Create a program to execute.  This should do nothing to the output.
		checkErr(AmetekDC_SetAutoSequenceRecordStart(DUT, channelName));

		// Program for testing will disabled for safety then switch between output 
		// set points.
	       checkErr( AmetekDC_ConfigureOutputEnabled( DUT, channelName, VI_FALSE ) );
		checkErr(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 6 ));
		checkErr(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 0 ));
		checkErr(AmetekDC_SetAutoSequenceRecordStop(DUT, channelName));		

		// Now the run the program to set the flag.
		checkErr(AmetekDC_SetAutoSequenceState(DUT, channelName,AMETEKDC_VAL_RUN));

	}
	else
	{
		// Clear the local lockout flag by making the mode local.
		checkErr(AmetekDC_SetAutoSequenceState(DUT, channelName, AMETEKDC_VAL_STOP));
	}  
   }
	

Error:
   // Restore the query status.
   AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, queryStatusState);
	   
   return( error );
}

/*****************************************************************************
 * Function: fnSetStatOperMeasFlag                                  
 * Purpose:  This function sets the SCPI operation status register, program running
 *              flag to the condition given by the bBitState parameter. This is 
 *              useful for setting up test of the status registers.  In addition this function 
 *              enables and disables the output while also causing changes to the set points.
 *
 * Note: This enables and disables the output, it should not be used with a load attached
 *         as the load could be damaged or the resulting flag might not be set!
 *
 *****************************************************************************/
ViStatus fnSetStatOperMeasFlag(ViSession DUT, 
                        ViChar channelName[], 
                        ViBoolean bBitState)
{
    ViStatus       error = VI_SUCCESS;
    ViReal64 measurement;
                                              
   if( !Ivi_Simulating( DUT ) )
   {
      ViSession io = Ivi_IOSession( DUT );

	// Select which sense to set the bit to.
	if(bBitState == VI_TRUE)
	{
		// Set the local lockout flag by making the mode remote.
		checkErr(AmetekDC_Measure( DUT, channelName, AMETEKDC_VAL_MEASURE_VOLTAGE, &measurement));		
	}
	else
	{
		// Clear the local lockout flag by making the mode local.
		checkErr(AmetekDC_Measure( DUT, channelName, AMETEKDC_VAL_MEASURE_VOLTAGE, &measurement));
	}  
   }
	

Error:
   return( error );
}
/*****************************************************************************
 * Function: fnTestSesr                                  
 * Purpose:  This function test each of the flags of the SESR register required
 *              by IEEE 488.2 standard (GPIB).
 *
 * NOTE: This test executes a reset causing parameters to be lost. Don't expect
 *           the same state after calling this function.
 *
 *
 *****************************************************************************/
ViStatus fnTestSesr(ViSession DUT, 
                        ViChar channelName[])
{
    ViStatus       error = VI_SUCCESS;
    ViInt32   errCode;
    ViChar tmpBuf[BUFFER_SIZE] = "";	
    ViChar channelAddr [ SMALL_BUFF_SIZE ];	
    ViBoolean bTestFailedState = VI_FALSE;	
    ViUInt16 uiRegValue = 0;
	ViBoolean queryStatusState = VI_TRUE;

    // Lock the session so that the the query status can be checked.
    checkErr( Ivi_LockSession( DUT, VI_NULL ) );
   
    // Back up the query and enable it.
    queryStatusState = Ivi_QueryInstrStatus(DUT);

   // Add the manditory multichannel address.
   if( ( strcmp( channelName, AMETEKDC_MASTER_CHANNEL ) == 0 ) || 
       ( strcmp( channelName, "a" ) == 0 ) )
   {
      // Using the master address.  Look up the bus address associate with the master address.
      AmetekDC_GetAttributeViString(DUT, VI_NULL, 
                                                 AMETEKDC_ATTR_BUS_ADDRESS,
                                                 SMALL_BUFF_SIZE,
                                                 channelAddr );
   }

   // For these tests, the device must be selected.
  sprintf(tmpBuf, "*ADR %s\n", channelAddr);
  checkErr(AmetekDC_WriteInstrData(DUT, tmpBuf));
  Delay(AMETEKDC_MIN_CMD_DELAY_SEC);
   
    // Test Operation Complete
    		// Send to verify operation complete.
    		checkErr(AmetekDC_WriteInstrData(DUT, "*OPC\n"));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);
		
		// Look for the standard Event register.
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_EVENT, &uiRegValue);
	
		if((uiRegValue & (1<<eSESR_OPC)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find an OPC flag!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;	
		}    
		
    // Test Query Error - Not used in XG, thus not tested.
		
    // Test Execution Error
    		// Cause a settings conflict error -221 by sending voltage out of range of OVP.
		// Set the voltage. Query for errors. There should be none.
            if(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 0 ) != VI_SUCCESS)
            	{
            		// Found issue, unable to configure the OVP set point. XG has logged the following errors:
            		AmetekDC_GetError( DUT, &error, BUFFER_SIZE, tmpBuf);

            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "SESR Test Failed: Failure to configure the set Vset to 0!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
            	}

		// Set OVP set point to 105% of Vset and query for errors.
            if(AmetekDC_ConfigureOVP( DUT,channelName, VI_TRUE, 6) != VI_SUCCESS)
            	{
            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to set OVPset. OVPset = 6!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
            	} 
			
		// Set the voltage. Query for errors. There should be none.
		AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, VI_FALSE);
             AmetekDC_ConfigureVoltageLevel( DUT,channelName, 6 );

		// Look for the standard Event register.
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_EVENT, &uiRegValue);

		if((uiRegValue & (1<<eSESR_EXE)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find an execution error!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;			
		}

		// Clear error queue and verify error code.
		checkErr(AmetekDC_error_query(DUT, &errCode, tmpBuf));
		
		if( errCode != -221 ) {
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find a settings conflict error! Error Code = %lf\n", errCode);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
		}
		
    // Test Command Error
    // Cause a Command Error -110 by sending an invalid string. 
        	// Send to verify operation complete.
  	      sprintf(tmpBuf, "*ADR %s;:COMMAND_ERROR_TEST\n", channelAddr);
    		checkErr(AmetekDC_WriteInstrData(DUT, tmpBuf));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);
			
		// Look for the standard Event register.
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_EVENT, &uiRegValue);

    		if((uiRegValue & (1<<eSESR_CME)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find a command error!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}
			
		// Clear error queue and verify error code.
		checkErr(AmetekDC_error_query(DUT, &errCode, tmpBuf));
		
		if( errCode != -110 ) {
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find a command error! Error Code = %lf\n", errCode);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
		}	
		
    // Test Device Dependent Error (none)
    // Test is done by causing the error queue to overflow.
		// By directly sending the rest command which takes upwards of 240 ms to complete.
		// we can follow up with a second write which will cause a buffer overrun. -363 error.
	      sprintf(tmpBuf, ":syst%s:res\n", channelAddr);
       	checkErr(AmetekDC_WriteInstrData(DUT, tmpBuf));
	      checkErr(AmetekDC_WriteInstrData(DUT, ":BUFFER_OVERRUN\n"));
	    // Delay long enough for the reset command to finish.
		Delay(0.5);
		
			// Look for the standard Event register.
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_EVENT, &uiRegValue);

    		if((uiRegValue & (1<<eSESR_DDE)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find a device dependent error!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}   
			
		// Clear error queue and verify error code.
		checkErr(AmetekDC_error_query(DUT, &errCode, tmpBuf));
		
		if( errCode != -363 ) {
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: Failed to find a buffer overrun error! Error Code = %lf\n", errCode);
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
		}				
		
    // Test PowerOn - Not used in XG, thus not tested.

    // Test summary bit.
    		//*Do negative test.  (Make sure propagation doesn't happen when summary disabled.
		// Disable the summary bits for SESR.	
		AmetekDC_SetRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_ENABLE, 0);
	
		// Send to verify operation complete.
  	      sprintf(tmpBuf, "*ADR %s;*OPC\n", channelAddr);
    		checkErr(AmetekDC_WriteInstrData(DUT, tmpBuf));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);

		// Look for the standard Event register.		
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STATUS_BYTE, &uiRegValue);
    		if((uiRegValue & (1<<eSBYTE_SESR)) != 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: SESR bit is set in status byte despit disabling the enable reg!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}   
			
		//*Do positive test to make sure that the summary bits work.	
		// Enable the summary bit for OPC.	
		AmetekDC_SetRegister(DUT, channelName, AMETEKDC_VAL_STAT_STANDARD_ENABLE, (1<<eSESR_OPC));
		
       	// Send to verify operation complete.
  	      sprintf(tmpBuf, "*ADR %s;*OPC;*ADR 0\n", channelAddr);
    		checkErr(AmetekDC_WriteInstrData(DUT, tmpBuf));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);		

		// Look for the standard Event register.		
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STATUS_BYTE, &uiRegValue);
    		if((uiRegValue & (1<<eSBYTE_SESR)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: SESR bit is clear in status byte despite enabling the SESR summary bits!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}  

		//* Test that clearing SESR also works with summary bits set.
		// Look for the standard Event register.		
		AmetekDC_QueryRegister(DUT, channelName,AMETEKDC_VAL_STAT_STANDARD_EVENT, &uiRegValue);

		// Look for the standard Event register.		
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STATUS_BYTE, &uiRegValue);
    		if((uiRegValue & (1<<eSBYTE_SESR)) != 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: SESR bit is set in status byte despite clearing the SESR register and having the summary bit enabled!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}  
			
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}   	

Error:
   // Restore the query status.
   AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, queryStatusState);
   
   return( error );
}


/*****************************************************************************
 * Function: fnTestStatusByte                                  
 * Purpose:  This function does a test of the Statys byte being set under different
 * 			fault conditions.
 *****************************************************************************/
ViStatus fnTestStatusByte(ViSession DUT, 
                        ViChar channelName[])
{
    ViStatus       error = VI_SUCCESS;
    ViChar tmpBuf[BUFFER_SIZE];	
    ViChar channelAddr [ SMALL_BUFF_SIZE ];	
    ViBoolean bTestFailedState = VI_FALSE;	
    ViUInt16 uiRegValue = 0;
	
    // Test error queue status flag.

    // Cause a settings conflict error -221 by sending voltage out of range of OVP.
		// Set the voltage. Query for errors. There should be none.
            if(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 1000 ) == VI_SUCCESS)
            	{
            		// Found issue, unable to configure the OVP set point. XG has logged the following errors:
            		AmetekDC_GetError( DUT, &error, BUFFER_SIZE, tmpBuf);

            		// Failure because this was a negative test.  We expect the feature to throw a settings conflict error.
	            sprintf( tmpBuf, "SESR Test Failed: Failure to configure the set Vset to 0!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;
            	}

		// Look for the standard Event register.		
		AmetekDC_QueryRegister(DUT, channelName, AMETEKDC_VAL_STAT_STATUS_BYTE, &uiRegValue);
    		if((uiRegValue & (1<<eSBYTE_ERR)) == 0)
		{
			// Failure because the correct error wasn't detected.
			// Test Failed, got the incorrect error.
	            sprintf( tmpBuf, "SESR Test Failed: SESR bit is set in status byte despite clearing the SESR register and having the summary bit enabled!\n");
					
              	bTestFailedState = VI_TRUE;
			goto Failed;				

		}  
	
    // Test SCPI Operation status summary bit - Summary bit is tested when the operation status register 
    // is tested.

    
    // Test SCPI Operation status summary bit - Summary bit is tested when the operation status register 
    // is tested.
    		
    // No need to test SESR summary bit as this is done in the SESR test function.
    
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	

Error:
   return( error );
}


	
/*****************************************************************************
 * Function: fnTestAutoSequence                                  
 * Purpose:  This function does a test all the function related to Autosequence programming.
 *
 *****************************************************************************/
ViStatus fnTestAutoSequence(ViSession DUT, 
                        ViChar channelName[])
{
    ViStatus       error = VI_SUCCESS;
    ViInt32   errCode;
    ViChar tmpBuf[BUFFER_SIZE] = "";	
    ViChar channelAddr [ SMALL_BUFF_SIZE ];	
    ViBoolean bTestFailedState = VI_FALSE;	
    ViInt32 bytesRead = 0;
    ViInt32 programSize = 0;
    ViBoolean queryStatusState = VI_TRUE;
    ViInt32 backupDwellTime = 0;
    ViInt32 backupRepeatCnt = 0;
	
    // Lock the session so that the the query status can be checked.
    checkErr( Ivi_LockSession( DUT, VI_NULL ) );
   
    // Back up the query and enable it.
    queryStatusState = Ivi_QueryInstrStatus(DUT);

    checkErr(AmetekDC_QueryAutoSequenceRepeat(DUT, channelName, tmpBuf));
    backupRepeatCnt = atoi(tmpBuf);

    checkErr(AmetekDC_QueryAutoSequenceDwellTime(DUT, channelName, &backupDwellTime));
	
   if( !Ivi_Simulating( DUT ) )
   {
      		ViSession io = Ivi_IOSession( DUT );
		
		// Set the local lockout flag by making the mode remote.
		checkErr(AmetekDC_SetAutoSequenceDwellTime(DUT, channelName, 1));
		checkErr(AmetekDC_SetAutoSequenceRepeat(DUT, channelName, 3));
		
             checkErr(AmetekDC_QueryAutoSequenceRepeat(DUT, channelName, tmpBuf));
		if(strcmp("3",tmpBuf ))
		{
			// Error found, the program is not made up of the expect number
			// of characters.
	            sprintf( tmpBuf, "AutoSequence Test Failed: Incorrect repeat count set, Repeat=%s!\n", tmpBuf);
						
	            	bTestFailedState = VI_TRUE;
			goto Failed;	
		}
		
		// Set the voltage. Query for errors. There should be none.
		AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, VI_FALSE);
			
		// Create a program to execute.  This should do nothing to the output.
		checkErr(AmetekDC_SetAutoSequenceRecordStart(DUT, channelName));

		// Program for testing will disabled for safety then switch between output 
		// set points.
	       checkErr(AmetekDC_ConfigureOutputEnabled( DUT, channelName, VI_FALSE ) );
		checkErr(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 6 ));
		checkErr(AmetekDC_ConfigureVoltageLevel( DUT,channelName, 0 ));
		checkErr(AmetekDC_SetAutoSequenceRecordStop(DUT, channelName));		

		// Calculate the expected program size. Account for the channel name size on a per command
		// basis.
		programSize = 14 + 17 + 17 + (3* strlen(channelName));
		
		// Check to make sure the sequence was correctly added.
		checkErr(AmetekDC_SendCmd(DUT, io, channelName, AMETEKDC_VAL_PROG_READBACK,"?"));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);
		
		//Read line. Noting that the unit will stop reading when a newline character is sent.
		AmetekDC_ReadInstrData(DUT, programSize, tmpBuf, &bytesRead);
		programSize -= bytesRead;
		
		//Read line. Noting that the unit will stop reading when a newline character is sent.
		AmetekDC_ReadInstrData(DUT, programSize, tmpBuf, &bytesRead);
		programSize -= bytesRead;	
		
		//Read line. Noting that the unit will stop reading when a newline character is sent.
		AmetekDC_ReadInstrData(DUT, programSize, tmpBuf, &bytesRead);
		programSize -= bytesRead;
		
		if(programSize != 0)
		{
			// Error found, the program is not made up of the expect number
			// of characters.
	            sprintf( tmpBuf, "AutoSequence Test Failed: The test program wasn't programmed correctly, there was %d bytes in the buffer!\n", bytesRead);
						
	            	bTestFailedState = VI_TRUE;
			goto Failed;			
		}
		
		// Now the run the program to set the flag.
		checkErr(AmetekDC_SetAutoSequenceState(DUT, channelName,AMETEKDC_VAL_RUN));
		checkErr(AmetekDC_QueryAutoSequenceState(DUT, channelName, tmpBuf));

		if(strcmp("RUNNING",tmpBuf))
		{
			// Error found, the program failed to start running.
	            sprintf( tmpBuf, "AutoSequence Test Failed: The program failed to trigger!\n");
						
	            	bTestFailedState = VI_TRUE;
			goto Failed;			
		}
		
		// Erase the string.
		*tmpBuf = 0;
		
		// Stop the Autosequence from running.
		checkErr(AmetekDC_SetAutoSequenceState(DUT, channelName, AMETEKDC_VAL_STOP));

		// Now test using the trigger method.
		checkErr(AmetekDC_SendAutoSequenceSoftwareTrigger(DUT, channelName));
		checkErr( AmetekDC_QueryAutoSequenceState(DUT, channelName, tmpBuf));

		if(strcmp("RUNNING",tmpBuf))
		{
			// Error found, the program failed to start running.
	            sprintf( tmpBuf, "AutoSequence Test Failed: The program failed to start!\n");
						
	            	bTestFailedState = VI_TRUE;
			goto Failed;			
		}

		// Stop the autosequence and delete it.
		checkErr(AmetekDC_SetAutoSequenceState(DUT, channelName, AMETEKDC_VAL_STOP));		
		checkErr(AmetekDC_DeleteAllSequences(DUT, channelName));

		// Check to make sure the sequence was deleted.
		checkErr(AmetekDC_SendCmd(DUT, io, channelName, AMETEKDC_VAL_PROG_READBACK,"?"));
		Delay(AMETEKDC_MIN_CMD_DELAY_SEC);
		AmetekDC_ReadInstrData(DUT, 2, tmpBuf, &bytesRead);
		
		if(bytesRead != 0)
		{
			// Error found, the program is not made up of the expect number
			// of characters.
      		      sprintf( tmpBuf, "AutoSequence Test Failed: The test program wasn't erased correctly, there was %d bytes in the buffer!\n", bytesRead);
					
	            	bTestFailedState = VI_TRUE;
			goto Failed;			
		}
   	}
   	
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	

Error:
   // Restore the dwell time and repeat count to the original values.
   AmetekDC_SetAutoSequenceRepeat(DUT, channelName, backupRepeatCnt);
   AmetekDC_SetAutoSequenceDwellTime(DUT, channelName, backupDwellTime);
		
   // Set the voltage. Query for errors. There should be none.
   AmetekDC_SetAttributeViBoolean(DUT, VI_NULL, IVI_ATTR_QUERY_INSTR_STATUS, queryStatusState);
   
   return( error );
}


/*****************************************************************************
 * Function: fnTestOVP                                  
 * Purpose:  This function does a test of the OVP at various voltages.  The
 *           intent is to verify that the calibration and the system works 
 *           correctly at all voltage ranges.
 *
 * NOTE: This function will enable the output and set voltages all the way up
 *       to the maximum model voltage.  Users of the function should take care
 *       to not have any sensative loads attached to the output during this test
 *       as it may result in damage to the load.
 *
 *****************************************************************************/
/*
ViStatus fnTestOVPRange(ViSession DUT, 
                        ViChar channelName[])
{
    ViStatus       error = VI_SUCCESS;
    ViInt32   errCode;
    ViChar tmpBuf[BUFFER_SIZE] = "";	
    ViChar channelAddr [ SMALL_BUFF_SIZE ];	
    ViBoolean bTestFailedState = VI_FALSE;	

			
Failed:
	if( bTestFailedState == VI_TRUE )
	{
#ifdef _CVI_
   		      MessagePopup( "%s", tmpBuf );
#else
		      printf( "%s", tmpBuf );
#endif   	
	}  
	

Error:
   return( error );
}
*/
